/* $Header: /lsid/semaphore/clarinet/custom/srcusrparb.c 1.6 1997/07/29 21:42:27 donm Exp $ */
/* Copyright (C) 1997, Hewlett-Packard Company, all rights reserved. */
/* Written by Don Mathiesen */

/*
This example program downloads srcusr.bin created from 56002 assembly
source template which is based on arb, then does a periodic arb,
with a partially different signal for each channel.
Works with sources on 1432/33/34.

Maximum of 5 source channels.

It includes examples for:
	* Download, select and run custom signal mode.
	* Store downloaded custom signal mode into user signal
		mode area of flash ROM.
	* Select and run signal mode from user area of flash ROM.
	* Erase custom signal mode from flash ROM.
	* Read source hw id register.
	* Read source status register.
	* Read source board parameters set at manufacture.
	* Read source rom firmware rev and date.
	* Read X memory on the fly.
*/

#include <stdlib.h>		/* For exit */
#include <stdio.h>		/* For printf */
#include <unistd.h>		/* For sleep */
#include <sicl.h>		/* For SICL functions */
#include <math.h>
#include "e1432.h"
#include "err1432.h"

#define	BLOCKSIZE	8192
#define	SIGLENGTH	8192
#define	SIGPERIOD	8192/16
#define	XFRLENGTH	4096
#define SRCBUFRSIZE	4096
#define MANUAL_ARM	1	/* 0 = autoarm/trig; 1 = manual arm/trig */

#define	SPAN		20000.0

#define	IN_MAX		32
#define	SRC_MAX		5

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

/* Make these global  */
static E1432ID hw;
static SHORTSIZ16 group;
static SHORTSIZ16 chan_list[IN_MAX + SRC_MAX];
static int chan_count;

static SHORTSIZ16 in_list[IN_MAX];
static SHORTSIZ16 in_group;
static int in_count;

static SHORTSIZ16 src_list[SRC_MAX];
static SHORTSIZ16 src_group;
static int src_count;

/*###########################################################*/
static int  trace = 0; /* 1=print 0= no print */


/******************* arb source routines **************************/

#define SIG_LENGTH_MAX 100000

static LONGSIZ32 src_sigbuff[SIG_LENGTH_MAX];
static LONGSIZ32 src_siglength;

static int src_sigperiod;
static SHORTSIZ16 arb_xfrmode, arb_xfrmodewait;

static SHORTSIZ16 arb_mode;
static LONGSIZ32 arb_srcbufsize;
static LONGSIZ32 arb_xfrsize;

/********* source command vectors and commands *********/
#define SRCCMDV_EXTMSG	0xa2a2a2a2UL
#define SRCCMDV_CMDMODE	0xa3a3a3a3UL
#define SRCCMDV_SRCON	0xB1B1B1B1UL
#define SRCCMDV_ROMID		0x98989898UL
#define SRCCMDV_ROMDATE		0x9E9E9E9EUL
#define SRCCMDV_READ_X		0xB7B7B7B7UL
#define SRCCMDV_READ_Y		0xB8B8B8B8UL

#define SRCCMD_BLOCKWRITE	0x0000000eUL
#define SRC_BLOCKWRITE_Y_SPACE	0x00000000UL
#define SRC_BLOCKWRITE_X_SPACE	0x00000001UL
#define SRC_BLOCKWRITE_P_SPACE	0x00000002UL
#define SRC_BLOCKWRITE_SIGADDR	0x00000068UL

#define SRCCMD_USERSTORE	0x0000000BUL
#define SRCCMD_USERCLR		0x0000000CUL
#define SRCCMD_USERCLR2		0x001D0001UL

#define SRCCMD_ACKMASK		0x00200000UL

#define	SRCCMD_BDPARMSRD	0x00000007UL
#define	SRCCMD_BDPARMSRESET	0x00000008UL

SHORTSIZ16 e1432_detect_modules(int maxlist, SHORTSIZ16 * lalist,
				SHORTSIZ16 * modulecount);


/*************************************************************/
/* can use e1432_src_rxfr instead of xfr_2src */
static int
xfr_2src(E1432ID lhw, SHORTSIZ16 chanID, LONGSIZ32 numwords, LONGSIZ32
	 *dptr, LONGSIZ32 mode)

{
    LONGSIZ32 i, words_left, xfrsize, maxtries;
    SHORTSIZ16 error;
    LONGSIZ32 xfrbuf_wds = 0;
    LONGSIZ32 srcbuf_state_A = 0;
    LONGSIZ32 srcbuf_state_B = 0;
    LONGSIZ32 srcbuf_state_AB = 0;

    if(trace)
    {
	(void)printf("\nxfr_2src: mode %x, numwords %d\n",mode,numwords);
	(void)printf("buf[0-3]: %x, %x, %x, %x\n",dptr[0],dptr[1],dptr[2],dptr[3]);
    }
    
    words_left = numwords;
    i = 0;
    maxtries = 10;
 
    while (words_left > 0)
    {
	if (words_left <= E1432_SRC_DATA_NUMWORDS_MAX)
	    xfrsize = words_left;
	else
	    xfrsize = E1432_SRC_DATA_NUMWORDS_MAX;

	/* write  xfrsize words to the substrate */
	if(trace)
	{
	    (void) printf("    e1432_write_srcbuffer_data() mode %d, xfrsize %d\n",mode,xfrsize);
	}
	
	error = e1432_write_srcbuffer_data(lhw, chanID,
					   dptr,
					   (SHORTSIZ16) xfrsize,
					   mode);
	if(trace)
	{
	    (void) printf("    e1432_write_srcbuffer_data returns: %d\n",error);
	}
	if (error == ERR1432_ACK_TIMEOUT)
	{
	    (void) printf("xfr_src_data: e1432_write_srcbuffer_data - timed out \n");
	    return error;
	}
	if (error != 0)
	{
	    i=i+1;
	    if (i == maxtries)
	    {
		(void) printf("xfr_src_data:    write with wait, timed out \n");	
		return error;
	    }
	    sleep (1);
	}
	else
	{
	    i=0;
	    dptr = dptr + xfrsize;
	    words_left = words_left - xfrsize;
	}
    }
    return 0;
}


/******************* arb source routines **************************/

#define SIG_LENGTH_MAX 100000

static LONGSIZ32 src_sigbuff1_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff2_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff3_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff4_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff5_a[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff1_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff2_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff3_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff4_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_sigbuff5_b[SIG_LENGTH_MAX];
static LONGSIZ32 src_siglength;

static int src_sigperiod;
static SHORTSIZ16 arb_xfrmode, arb_xfrmodewait;

static SHORTSIZ16 arb_mode;
static LONGSIZ32 arb_srcbufsize;
static LONGSIZ32 arb_xfrsize;

void
src_sig( LONGSIZ32 siglength, int sigperiod, long *sigbuff)
{

int i;

	  /* Synth a sine */
          for(i=0;i<siglength;i++)
	  {
             sigbuff[i] = (long)((float) 0x7fffffffL *
                          sin(2.0*M_PI*(float)i/((float)sigperiod)));
          }
}

void
xfr_src_sig(E1432ID lhw, SHORTSIZ16 chanID, long *sigbuff, int xfrlength, SHORTSIZ16 mode)
{
    LONGSIZ32 i, maxtries;
    short error;
    SHORTSIZ16 status;
    LONGSIZ32 xfrbuf_wds,srcbuf_state_AB,srcbuf_state_A,srcbuf_state_B;
   
	/* write xfrlength words to the substrate */

	i = 0;
	maxtries = 10;

	for (i = 0; i < maxtries; i++)
	{
#ifdef DPRINT2
	    e1432_get_src_arbstates(lhw, chanID, &xfrbuf_wds, &srcbuf_state_AB, &srcbuf_state_A, &srcbuf_state_B);
	    (void) printf("xfrbuf_wds %d, srcbuf_state_AB 0x%x, srcbuf_state_A 0x%x, srcbuf_state_Bxf 0x%x\n",xfrbuf_wds,srcbuf_state_AB,srcbuf_state_A, srcbuf_state_B);
#endif	    
	    status = e1432_check_src_arbrdy(lhw, chanID, mode);
#ifdef DPRINT2
	    (void) printf("xfr_src_sig: status 0x%x\n",status);
#endif
	    
	    if (status == 1)
	    {
		error = e1432_write_srcbuffer_data(lhw, chanID,
						  sigbuff,
						  (SHORTSIZ16)  xfrlength,
						  mode);
		if(error)
		{
		    (void) printf("Error %d in e1432_write_srcbuffer_data\n",
				  error);
		    e1432_get_src_arbstates(lhw, chanID, &xfrbuf_wds, &srcbuf_state_AB, &srcbuf_state_A, &srcbuf_state_B);
		    (void) printf("xfrbuf_wds %d, srcbuf_state_AB 0x%x, srcbuf_state_A 0x%x, srcbuf_state_Bxf 0x%x\n",xfrbuf_wds,srcbuf_state_AB,srcbuf_state_A, srcbuf_state_B);
	    
		    status = e1432_check_src_arbrdy(lhw, chanID, mode);
		    (void) printf("xfr_src_sig: status 0x%x\n",status);
		}
#ifdef DPRINT2
		e1432_get_src_arbstates(lhw, chanID, &xfrbuf_wds, &srcbuf_state_AB, &srcbuf_state_A, &srcbuf_state_B);
		(void) printf("xfrbuf_wds %d, srcbuf_state_AB 0x%x, srcbuf_state_A 0x%x, srcbuf_state_Bxf 0x%x\n",xfrbuf_wds,srcbuf_state_AB,srcbuf_state_A, srcbuf_state_B);
	    
		status = e1432_check_src_arbrdy(lhw, chanID, mode);
		(void) printf("xfr_src_sig: status 0x%x\n",status);
#endif
		return;
	    }
	}
	if (i == maxtries)
	{
	    (void) printf("xfr_src_sig: timed out \n");
	    e1432_get_src_arbstates(lhw, chanID, &xfrbuf_wds, &srcbuf_state_AB, &srcbuf_state_A, &srcbuf_state_B);
	    (void) printf("xfrbuf_wds %d, srcbuf_state_AB 0x%x, srcbuf_state_A 0x%x, srcbuf_state_Bxf 0x%x\n",xfrbuf_wds,srcbuf_state_AB,srcbuf_state_A, srcbuf_state_B);
	    
	    status = e1432_check_src_arbrdy(lhw, chanID, mode);
	    (void) printf("xfr_src_sig: status 0x%x\n",status);
	    return;
	}
}

/***************** end arb source routines ************************/

static int
read_data(void)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    int     i;

    (void) printf("read_data\n");
    /* Read some data */
    for (i = 0; i < in_count; i++)
    {
	CHECK(e1432_read_float64_data(hw, in_list[i],
				      E1432_TIME_DATA, buffer,
				      BLOCKSIZE, NULL, &count));
	if(trace)
	{
	    (void) printf("channel %d    data points read =  %d\n", i , count);
	}
	
	if (count != BLOCKSIZE)
	{
	    DEBUG((void) printf("Actual count was %d\n", count));
	    return -1;
	}
    }
    return 0;
}

int
main(void)
{

/*###########################################################*/
/* custom signal mode examples: */

    int read_romid = 1;  /* 1= read ROM id and date, 0= don't read */
    int read_bdparms = 1;  /* 1= read bd parms set durring manufacture, 0= don't read */
    int read_status = 1;  /* 1= read, 0= don't read */
    int sig_dnld = 1; /* 1= download, 0= user rom area */

    /* *_userrom REQUIRES source rom firmware 961209 or later */
    int store_in_userrom = 0;  /* 1= store and use, 0= don't store */
    int erase_userrom = 0;  /* 1= erase, 0= don't erase */

    int runmeas = 1;  /* 1= run measurement, 0= don't run */

    int sig20bit = 0;    /* 1 = 20 bit mode(6.4 kHz max sig bw,single channel,
			 0 = 16 bit mode (25.6KHz bw, 2 channel for E1434)
			 span should be chosen for proper sample rate,
			 Fs = 2.56 Fspan
			 */
    
    int     i, error, xfr_error,k;
    struct e1432_hwconfig hwconfig[15];
    SHORTSIZ16 laddr = 8;
    int wait, nowait, manual_arm, ln, lw;
    int doitforever;
    
    LONGSIZ32 trapDownload = 0x0c0000;  /* jmp <0 */

    LONGSIZ32 *dptr = &trapDownload;
    LONGSIZ32 dsize = sizeof(trapDownload);
    LONGSIZ32 bufr[256];
    int bufrindx;
    char *file_name = "srcusr.bin";
    off_t file_size;
    int file_des;
    int la_override = 0;

    FLOATSIZ64 startTime, currentTime;

/* bdparms set at manufacture */
struct bdpstruct1 {
  unsigned structrevid; /* revid of this structure (0x000001) */
  unsigned hwrevid;     /* revid of hardware (0x000001) */
  unsigned date[12];    /* date last changed  1char per word */
  unsigned corr[16];    /* correction factors */
  unsigned unused1[60-30];      /* unused to fill first 60 words */
  unsigned hwrev_1[16];     /* hardware revision character string 1char per word */
  unsigned comments[44];    /* comments string  1char per word */
} bdparms1;

    unsigned int * bdparms1ptr;
    char bdparmstring[120];
    int readxdata_num,readxdata_addr;
 
    SHORTSIZ16 modulecount;
    SHORTSIZ16 modulelist[15];
    SHORTSIZ16 src_ch_state[5];
    LONGSIZ32 *src_sigbuffptrs_a[5];   
    LONGSIZ32 *src_sigbuffptrs_b[5];   
    LONGSIZ32 arb_xfrsize2,arb_srcbufleft;
    LONGSIZ32 arb_srcbufsize2;
    src_sigbuffptrs_a[0]= &src_sigbuff1_a[0];   
    src_sigbuffptrs_a[1]= &src_sigbuff2_a[0];   
    src_sigbuffptrs_a[2]= &src_sigbuff3_a[0];   
    src_sigbuffptrs_a[3]= &src_sigbuff4_a[0];   
    src_sigbuffptrs_a[4]= &src_sigbuff5_a[0];   
    src_sigbuffptrs_b[0]= &src_sigbuff1_b[0];   
    src_sigbuffptrs_b[1]= &src_sigbuff2_b[0];   
    src_sigbuffptrs_b[2]= &src_sigbuff3_b[0];   
    src_sigbuffptrs_b[3]= &src_sigbuff4_b[0];   
    src_sigbuffptrs_b[4]= &src_sigbuff5_b[0];   

    readxdata_addr = 0x5010;
    readxdata_num = 64; /* read x data on the fly */

    wait = 1;
    nowait = 0;
    manual_arm = MANUAL_ARM;

    /* Initialize the library */
    CHECK(e1432_init_io_driver());

    /* Change this 0 to 1 to see call tracing */
    e1432_trace_level(0);

    error = e1432_detect_modules(15, modulelist, &modulecount);
    if (error)
    {
	printf("detect_modules failed\n");
	exit(1);
    }

    /* Use e1432_get_hwconfig to see if the module already has
       firmware.  If this errors, assume we need to install firmware,
       so use e1432_install to do it. */

    for (i = 0; i < modulecount; i++)
    {
	(void) e1432_print_errors(0);
	error = e1432_get_hwconfig(1, &(modulelist[i]), &(hwconfig[i]));
	if (error != 0)
	{
	    (void) printf("downloading /opt/e1432/lib/sema.bin to %d\n",(int)(modulelist[i]));
	    (void) fflush(stdout);
	    error = e1432_install(1, &(modulelist[i]), 0, "/opt/e1432/lib/sema.bin");
	    if (error)
	    {
		(void) printf("e1432_install failed, error: %d\n", error);
		return -1;
	    }
	
	    CHECK(e1432_get_hwconfig(1, &(modulelist[i]), &(hwconfig[i])));
	}
    }
    
    /* Turn on debugging prints, a good idea while developing.  But do
       it after the above e1432_get_hwconfig, because we know that
       might error if the firmware is not yet installed in the
       module. */
    (void) e1432_print_errors(1);

    e1432_debug_level(0);
    CHECK(e1432_assign_channel_numbers(modulecount, modulelist, &hw));

    in_count = 0;
    src_count = 0;
    for (i = 0; i < modulecount; i++)
    {
	in_count += hwconfig[i].input_chans;
	src_count += hwconfig[i].source_chans;
	printf("E1432 at LA %d has %d input channels\n", modulelist[i],
	       hwconfig[i].input_chans);
 	printf("E1432 at LA %d has %d source channels\n", modulelist[i],
	       hwconfig[i].source_chans);
    }

    /* Create input channel group */
    if ( in_count > 0)
    {
	if (in_count > IN_MAX)
	    in_count = IN_MAX;
	for (i = 0; i < in_count; i++)
	    in_list[i] = E1432_INPUT_CHAN(i+1);
	in_group = e1432_create_channel_group(hw, in_count, in_list);
	
	if (in_group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
				in_group));
	    return -1;
	}
    }
    else
    {
	in_count =0;
    }

    /* Create source group */
    if ( src_count > 0)
    {
	if (src_count > SRC_MAX)
	    src_count = SRC_MAX;
	k=0;
	for (i = 0; i < src_count; i++)
	{
	    /* with 1434 in slot 4, LA 9, and 1432 in slot 8, LA 8, */
	    if (i == 0) src_list[k++] = E1432_SOURCE_CHAN(i+1); /* 1434 ch 1 */
	    if (i == 1) src_list[k++] = E1432_SOURCE_CHAN(i+1); /* 1434 ch 2 */
	    if (i == 2) src_list[k++] = E1432_SOURCE_CHAN(i+1); /* 1434 ch 3 */
	    if (i == 3) src_list[k++] = E1432_SOURCE_CHAN(i+1); /* 1434 ch 4 */
	    if (i == 4) src_list[k++] = E1432_SOURCE_CHAN(i+1); /* 1434 ch 5 */
	}
	src_count =k;
	src_group = e1432_create_channel_group(hw, src_count, &src_list[0]);
	if (src_group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
			    src_group));
	    return -1;
	}
    }
    else
    {
	src_count =0;
	(void) printf("Error : No source channels found\n");
	/* exit(1); */
    }

    /* Create global channel group */
    chan_count = 0;
    for (i = 0; i < in_count; i++)
    {
	chan_list[chan_count] = in_list[i];
	chan_count = chan_count+1;
    }
    for (i = 0; i < src_count; i++)
    {
	chan_list[chan_count] = src_list[i];
	chan_count = chan_count+1;
    }
    if (chan_count > 0)
    { 
	group = e1432_create_channel_group(hw, chan_count, chan_list);
	if (group >= 0)
	{
	    DEBUG((void) printf("e1432_create_channel_group returned %d\n",
				group));
	    exit(1);
	}
    }

/* initialize all channels */
    CHECK(e1432_set_clock_freq(hw, group,51200));
    CHECK(e1432_set_span(hw, group, SPAN));
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));

    (void) printf("manual_arm (1=manual, 0 = auto) %d\n",manual_arm);
    if (manual_arm)
    {
	CHECK(e1432_set_auto_trigger(hw, group, E1432_MANUAL_TRIGGER));
	CHECK(e1432_set_auto_arm(hw, group, E1432_MANUAL_ARM));
    }
    else
    {
	CHECK(e1432_set_auto_trigger(hw, group, E1432_AUTO_TRIGGER));
	CHECK(e1432_set_auto_arm(hw, group, E1432_AUTO_ARM));
    }

/* initialize inputs */
    CHECK(e1432_set_active(hw, in_group, E1432_CHANNEL_ON));

    if(sig_dnld)
    {
/*
******** Open and read download source code file ********
*/

	if ( file_name != NULL )
	{
	    CHECK(i1432_open_file(file_name, &file_des, &file_size));
	    dsize = file_size;
	    (void)printf("filename = %s, file_size = %d\n",file_name,file_size);
	    dptr = (LONGSIZ32 *) malloc(file_size);
	    if (dptr == NULL)
	    {
		(void) fprintf(stderr,"malloc failure\n");
		exit(2);
	    }
	    if (read(file_des, dptr, file_size) != (int) file_size)
	    {
		(void) fprintf(stderr,"file read failure\n");
		exit(2);
	    }
	}
	else
	{
	    (void)printf("file srcusr.bin not found in local directory\n");
	    return(1);
	}
/*
********* Shift download source code *********
*/
	/* shift words left since only top 3 bytes sent to 56000 */
	for (i=0; i<file_size/4; i++) dptr[i]= dptr[i] << 8;
    }    
/*
******** Initialize transfer buffer for the source channels ********
*/
    CHECK(e1432_set_active(hw, src_group, E1432_CHANNEL_ON));

    CHECK(e1432_set_srcbuffer_init(hw, src_group,E1432_SRCBUFFER_INIT_XFER));

    /*************************************************************/
    /* for all channel in the list */   
    for (i = 0; i < src_count; i++)
    {

	if(read_romid)
	{
/*
******** read rom id and date ********
*/
/* This step is not needed for the download but is here as an example */
	    bufr[0] = SRCCMDV_CMDMODE;
	    xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
				 E1432_SRC_DATA_MODE_RAWCMD);

	    bufr[0] = SRCCMDV_ROMID;
	    xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
				 E1432_SRC_DATA_MODE_RAWCMD);

	    bufr[0] = 0;
	    bufr[1] = 0;
	    xfr_error = xfr_2src(hw, src_list[i], 2, bufr,
		     E1432_SRC_DATA_MODE_RAWREAD);
	    (void)printf("source ROM ID:  %x hex, %d dec\n",(bufr[1]>>8 & 0x00FFFFFF),(bufr[1]>>8 & 0x00FFFFFF));

	    bufr[0] = SRCCMDV_ROMDATE;
	    xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
				 E1432_SRC_DATA_MODE_RAWCMD);

	    bufr[0] = 0;
	    bufr[1] = 0;
	    xfr_error = xfr_2src(hw, src_list[i], 2, bufr,
		     E1432_SRC_DATA_MODE_RAWREAD);
	    (void)printf("source ROM date: %x\n",(bufr[1]>>8 & 0x00FFFFFF));
	}

	if(read_bdparms)
	{
/*
******** read board parameters set during manufacture ********
*/
	    bufr[0] = SRCCMDV_CMDMODE;
	    xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
				 E1432_SRC_DATA_MODE_RAWCMD);
    /* must read in 2 blocks of 60 (+1 garbage) words due to CMDMODE
       dsp host port data buffer 64 wd limits */
	    bdparms1ptr = (unsigned int *) &bdparms1;

    /* read bd parms block 1 */
	    bufr[0] = SRCCMD_BDPARMSRD<<8;
	    bufr[1] = 1<<8;
	    xfr_error = xfr_2src(hw, src_list[i], 2, bufr,
				 E1432_SRC_DATA_MODE_RAWDATA);
	    bufr[0] = SRCCMDV_EXTMSG;
	    xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
				 E1432_SRC_DATA_MODE_RAWCMD);
	    bufr[0] = 0;
	    bufr[1] = 0;
	    xfr_error = xfr_2src(hw, src_list[i], 61, bufr,
		     E1432_SRC_DATA_MODE_RAWREAD);
	    for (bufrindx = 0; bufrindx < 61; bufrindx++)
	    { 
/*
		(void)printf("    bufr [%d] = %x, %.4c\n",bufrindx+1,bufr[bufrindx+1],bufr[bufrindx+1]);
*/
	    /* skip first word as it is garbage */
		*bdparms1ptr++ = bufr[bufrindx+1]>>8 & 0x00FFFFFF;
	    }

     /* read bd parms block 2 */
	    bufr[0] = SRCCMD_BDPARMSRD<<8;
	    bufr[1] = 2<<8;
	    xfr_error = xfr_2src(hw, src_list[i], 2, bufr,
				 E1432_SRC_DATA_MODE_RAWDATA);
	    bufr[0] = SRCCMDV_EXTMSG;
	    xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
				 E1432_SRC_DATA_MODE_RAWCMD);
	    bufr[0] = 0;
	    bufr[1] = 0;
	    xfr_error = xfr_2src(hw, src_list[i], 61, bufr,
		     E1432_SRC_DATA_MODE_RAWREAD);
	    for (bufrindx = 0; bufrindx < 61; bufrindx++)
	    { 
/*
	    (void)printf("    bufr [%d] = %x, %.4c\n",bufrindx+1,bufr[bufrindx+1],bufr[bufrindx+1]);
*/
	    /* skip first word as it is garbage */
		*bdparms1ptr++ = bufr[bufrindx+1]>>8 & 0x00FFFFFF;
	    }

	    bdparms1ptr = (unsigned int *) &bdparms1;
	    bdparms1ptr++;

	    (void)printf("\nsource board parameters for source = %x\n",src_list[i]);
	    (void)printf("    revid of hardware: %x hex, %d dec\n",*bdparms1ptr,*bdparms1ptr);
	    /* put 12 words into string */
	    for (bufrindx = 0; bufrindx < 12; bufrindx++)
	    {
		bdparmstring[bufrindx] = *bdparms1ptr++;
	    }
	    bdparmstring[bufrindx] = 0x0;
	    
	    (void)printf("    date last changed char [12]: %s\n",bdparmstring);
	}
	
	if(read_status)
	{
/*
******** read id and sisr status ********
*/
/* This step is not needed for the download but is here as an example */
	    bufr[0] = 0;
	    xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
		     E1432_SRC_DATA_MODE_READID);
	    (void)printf("source ID reg: %x\n",bufr[0]);
	    bufr[0] = 0;
	    xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
		     E1432_SRC_DATA_MODE_READSTATUS);
	    (void)printf("source SISR reg: %x\n",bufr[0]);
	}
	
	if(sig_dnld)
	{
/*
******** send cmd to put source in hostport in command buffer mode ********
*/
	    (void)printf("***starting source signal mode download process\n");
	    (void)i1432_get_time(&startTime);
	    bufr[0] = SRCCMDV_CMDMODE;
	    xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
				 E1432_SRC_DATA_MODE_RAWCMD);
	
/*
******** send cmd to to clear signal mode ********
*/
	    CHECK(e1432_set_source_mode(hw,src_list[i], E1432_SOURCE_MODE_CLR));

/*
******** send extended message (data and cmd) to set up program transfer ********
*/
	    bufr[0] = SRCCMD_BLOCKWRITE<<8;
	    bufr[1] = SRC_BLOCKWRITE_P_SPACE<<8;
	    bufr[2] = SRC_BLOCKWRITE_SIGADDR<<8;
	    xfr_error = xfr_2src(hw, src_list[i], 3, bufr,
				 E1432_SRC_DATA_MODE_RAWDATA);
	    bufr[0] = SRCCMDV_EXTMSG;
	    xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
				 E1432_SRC_DATA_MODE_RAWCMD);

/*
******** transfer data ********
*/
	    xfr_error = xfr_2src(hw, src_list[i], dsize/4,dptr,E1432_SRC_DATA_MODE_RAWDATA);

/*
******** end transfer by placing hostport in command buffer mode  ********
*/
	    bufr[0] = SRCCMDV_CMDMODE;
	    xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
				 E1432_SRC_DATA_MODE_RAWCMD);
	    (void)i1432_get_time(&currentTime);
	    (void)printf("***source signal mode download done, time = %f seconds\n",currentTime-startTime);
	}
	
/*
******** store signal mode in user area of flash ROM ********
*/
	if(store_in_userrom)
	{
	    (void)printf("**** storing download in user area of ROM\n");
	    sig_dnld = 0;
/*
******** send extended message (data and cmd) to store signal mode in flash ROM ********
*/
/******** store erases then writes to flash ROM  **********************/

/* Timeout error if E1432_SRC_DATA_MODE_RAWCMD (which waits for ack) is
   used.  Therefore using _RAWNACKCMD which does not wait, then check
   status with nonclearing status read, _READNCLRSTATUS, and ending
   by clearing status with _READSTATUS */

	    bufr[0] = SRCCMD_USERSTORE<<8;
	    bufr[1] = SRCCMD_USERCLR2<<8;
	    xfr_error = xfr_2src(hw, src_list[i], 2, bufr,
		     E1432_SRC_DATA_MODE_RAWDATA);
	    bufr[0] = SRCCMDV_EXTMSG;
	    (void)i1432_get_time(&startTime);
	    xfr_error = xfr_2src(hw, src_list[i], 1, &bufr[0],
		     E1432_SRC_DATA_MODE_RAWNACKCMD);

    /* wait for completion */
	    xfr_error =0;
	    (void)printf("wait for completion\n");
	    bufr[0] = 0;
	    xfr_error = xfr_2src(hw, src_list[i], 1, &bufr[0],
				 E1432_SRC_DATA_MODE_READNCLRSTATUS);
	    (void)i1432_get_time(&currentTime);
	    (void)printf("Flash ROM programming time = %f seconds\n",currentTime-startTime);
	    (void)printf("source SISR reg: %x\n",bufr[0]);
	    while(((bufr[0] & SRCCMD_ACKMASK) == 0) && (currentTime-startTime < 60))
	    {
		bufr[0] = 0;
		xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
				     E1432_SRC_DATA_MODE_READNCLRSTATUS);
		(void)printf("source SISR reg: %x\n",bufr[0]);
		(void)i1432_get_time(&currentTime);
		(void)printf("Flash ROM programming time = %f seconds\n",currentTime-startTime);
	    }
	    (void)printf("Flash ROM programming DONE time = %f seconds\n",currentTime-startTime);
	    /* clear status */
		xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
				     E1432_SRC_DATA_MODE_READSTATUS);
		(void)printf("source SISR reg: %x\n",bufr[0]);
	    
	}

	if(erase_userrom)
	{
	    (void)printf("**** erasing user area of ROM\n");
/*
******** send extended message (data and cmd) to erase signal mode in flash ROM ********
*/

/* if user rom signal mode is selected but it is erased, the relays click
   but no signal is output and it is terminated, select another signal mode */

	    bufr[0] = SRCCMD_USERCLR<<8;
	    bufr[1] = SRCCMD_USERCLR2<<8;
	    xfr_error = xfr_2src(hw, src_list[i], 2, bufr,
		     E1432_SRC_DATA_MODE_RAWDATA);
	    bufr[0] = SRCCMDV_EXTMSG;
	    xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
		     E1432_SRC_DATA_MODE_RAWNACKCMD);

    /* wait for completion */
	    xfr_error =0;
	    (void)printf("wait for completion\n");
	    bufr[0] = 0;
	    xfr_error = xfr_2src(hw, src_list[i], 1, &bufr[0],
				 E1432_SRC_DATA_MODE_READNCLRSTATUS);
	    (void)i1432_get_time(&currentTime);
	    (void)printf("Flash ROM programming time = %f seconds\n",currentTime-startTime);
	    (void)printf("source SISR reg: %x\n",bufr[0]);
	    while(((bufr[0] & SRCCMD_ACKMASK) == 0) && (currentTime-startTime < 60))
	    {
		bufr[0] = 0;
		xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
				     E1432_SRC_DATA_MODE_READNCLRSTATUS);
		(void)printf("source SISR reg: %x\n",bufr[0]);
		(void)i1432_get_time(&currentTime);
		(void)printf("user Flash ROM erase time = %f seconds\n",currentTime-startTime);
	    }
	    (void)printf("user Flash ROM erased DONE time = %f seconds\n",currentTime-startTime);
	    /* clear status */
		xfr_error = xfr_2src(hw, src_list[i], 1, bufr,
				     E1432_SRC_DATA_MODE_READSTATUS);
		(void)printf("source SISR reg: %x\n",bufr[0]);
	    
	}


    }   /* end operations to all source channels in the list */

    (void)printf("\n");
    
    if (!runmeas)
    {
	printf("not running measurement\n");
	return 0;
    }
    
/* initialize source */
    if (src_count > 0)
    {

	arb_mode = E1432_SRCBUFFER_PERIODIC_AB;
/*
	arb_mode = E1432_SRCBUFFER_PERIODIC_A;
*/

	if ((arb_mode == E1432_SRCBUFFER_PERIODIC_AB)||(arb_mode == E1432_SRCBUFFER_CONTINUOUS))
	{
	    arb_xfrmode = E1432_SRC_DATA_MODE_AB;
	    arb_xfrmodewait = E1432_SRC_DATA_MODE_WAITAB;
	}
	else
	{
	    arb_xfrmode = E1432_SRC_DATA_MODE_A;
	    arb_xfrmodewait = E1432_SRC_DATA_MODE_WAITA;
	}
	
	
	arb_srcbufsize = SRCBUFRSIZE;
	src_siglength = SIGLENGTH;
	src_sigperiod = SIGPERIOD;
	
	arb_xfrsize = XFRLENGTH;

	(void) printf("arb_xfrsize %d, arb_srcbufsize %d, src_siglength %d\n",
		      arb_xfrsize, arb_srcbufsize, src_siglength);
	(void) printf("arb_xfrmode 0x%x, arb_xfrmodewait %d\n",
		      arb_xfrmode, arb_xfrmodewait);

	CHECK(e1432_set_active(hw, src_group, E1432_CHANNEL_ON));

    if (manual_arm)
    {
	CHECK(e1432_set_data_mode(hw, in_group, E1432_BLOCK_MODE));

/*
******** selects burst signal mode ***********
*/
	if(sig_dnld)
	{
	/* download signal mode,  initializes the downloaded signal mode */
	    (void)printf("**** using signal mode downloaded\n");
	    CHECK(e1432_set_source_mode(hw, src_group,
					E1432_SOURCE_MODE_BDNLD));
	}
	else
	{
	    /* user ROM signal mode, overwrite signal mode from user ROM and initialize it */
	    (void)printf("**** using signal mode from user area of ROM\n");
	    CHECK(e1432_set_source_mode(hw, src_group, E1432_SOURCE_MODE_BUSR));
	}
	CHECK(e1432_set_duty_cycle(hw, src_group, 1.0));
	if (arb_xfrmode == E1432_SRC_DATA_MODE_AB)
	{
	    CHECK(e1432_set_source_blocksize(hw, src_group, 2*SRCBUFRSIZE));
	}
	else
	{
	    CHECK(e1432_set_source_blocksize(hw, src_group, SRCBUFRSIZE));
	}
    }
    else
    {
	CHECK(e1432_set_data_mode(hw, in_group, E1432_CONTINUOUS_MODE));

/*
******** selects continuous signal mode ***********
*/
	if(sig_dnld)
	{
	    /* download signal mode */
	    (void)printf("**** using signal mode downloaded\n");
	    CHECK(e1432_set_source_mode(hw, src_group,
					E1432_SOURCE_MODE_DNLD));
	}
	else
	{
	    /* user ROM signal mode */
	    (void)printf("**** using signal mode from user area of ROM\n");
	    CHECK(e1432_set_source_mode(hw, src_group, E1432_SOURCE_MODE_USR));
	}

	CHECK(e1432_set_source_blocksize(hw, src_group, SRCBUFRSIZE));
    }

	CHECK(e1432_set_ramp_rate(hw, src_group, 0));
	CHECK(e1432_set_range(hw, src_group, 1.0));
/*	CHECK(e1432_set_span(hw, src_group, 625)); */
	CHECK(e1432_set_amp_scale(hw, src_group, 1.0));

	CHECK(e1432_auto_zero(hw, src_group));

	CHECK(e1432_set_srcbuffer_mode(hw, src_group, arb_mode));
	CHECK(e1432_set_srcbuffer_size(hw, src_group, arb_srcbufsize));
	CHECK(e1432_set_srcbuffer_init(hw, src_group,
				       E1432_SRCBUFFER_INIT_EMPTY));

/** set up 20 bit path **/
	/* signal bandwidth should be less than 6.4 kHz,
	   span should be chosen for proper sample rate,
	   Fs = 2.56 Fspan */
			 
	/* setting E1432_ANTI_ALIAS_DIGITAL_OFF sets the 20 bit single
	   channel path, setting the _filter_freq to 6400.0, filters out
	   the aliases. */

	/* These calls should be made AFTER active source channels are set,
	   and AFTER the signal mode is set. */

	if(sig20bit)
	{
	    (void) printf("20 bit mode\n");
	    CHECK(e1432_set_anti_alias_digital(hw, src_group,
						E1432_ANTI_ALIAS_DIGITAL_OFF));
	    CHECK(e1432_set_filter_freq(hw, src_group,6400.00));
	}


	/** create signals ***/
	for (i=0; i<src_count;i++)
	{	
	    src_sig(src_siglength, 2*src_sigperiod/(2*(i+1)), src_sigbuffptrs_a[i]);
	    src_sig(src_siglength, 2*src_sigperiod, src_sigbuffptrs_b[i]);
	}
	
	/* initialize source buffers  */
	for (i=0; i<src_count;i++)
	{	
	    CHECK(e1432_set_srcbuffer_mode(hw, src_list[i], arb_mode));
	    CHECK(e1432_set_srcbuffer_size(hw, src_list[i], arb_srcbufsize));
	    CHECK(e1432_set_srcbuffer_init(hw, src_list[i],
					   E1432_SRCBUFFER_INIT_EMPTY));
	    CHECK(e1432_get_srcbuffer_size(hw, src_list[i],
					   &arb_srcbufsize2));
	    if (arb_srcbufsize2 != arb_srcbufsize)
	    {
		printf("Source buffer size changed from %d to %d \n",
		       arb_srcbufsize,arb_srcbufsize2);
		
		arb_srcbufsize = arb_srcbufsize2;
	    }
	}	
	/* down load arb source buffer A signal */
	/* (void) printf("preload buffer A\n"); */
	arb_xfrsize2 = arb_xfrsize;
	for (k=0; k<arb_srcbufsize;k = k + arb_xfrsize)
	{	
	    arb_srcbufleft = arb_srcbufsize - k;
	    if (arb_srcbufleft < arb_xfrsize) arb_xfrsize2 = arb_srcbufleft;
	    for (i=0; i<src_count;i++)
	    {	
		(void) printf("preload buffer A, chan %x, %d words\n",src_list[i],arb_xfrsize2);
		xfr_src_sig(hw, src_list[i], src_sigbuffptrs_a[i]+k,arb_xfrsize2,arb_xfrmodewait);
	    }
	}
	if (arb_xfrmode == E1432_SRC_DATA_MODE_AB)
	{
	    /* (void) printf("preload buffer B\n"); */
	    /* down load arb source buffer B signal */
	    arb_xfrsize2 = arb_xfrsize;
	    for (k=0; k<arb_srcbufsize;k = k + arb_xfrsize)
	    {	
		arb_srcbufleft = arb_srcbufsize - k;
		if (arb_srcbufleft < arb_xfrsize) arb_xfrsize2 = arb_srcbufleft;
		for (i=0; i<src_count;i++)
		{	
		    (void) printf("preload buffer B, chan %x, %d words\n",src_list[i],arb_xfrsize2);
		    
		    xfr_src_sig(hw, src_list[i],
				    src_sigbuffptrs_b[i]+k,arb_xfrsize2,arb_xfrmodewait);
		    
		}
	    }
	}
	(void) printf("done pre-loading buffer(s)\n");
    }
    else
    {
	(void) printf("Error : No source channels found\n");
	/* exit(1); */
    }

/* Start measurement */
    (void) printf("init_measure\n\n");
    CHECK(e1432_init_measure(hw, group));
    


    if ((readxdata_num>0)&& (!manual_arm))
    {
	/* set up to read x memory */
	/* must be after init_measure, and arm, which disables this mode */
	bufr[0] = SRCCMDV_READ_X;
/*	(void) printf("send SRCCMDV_READ_X:  %x\n",bufr[0]); */
	xfr_error = xfr_2src(hw, src_list[0], 1, bufr,
			     E1432_SRC_DATA_MODE_RAWCMD);
    }

    doitforever = 1;
    
    while(doitforever)
    {
	if (manual_arm)
	{
	    (void) printf("arm_measure \n");
	    CHECK(e1432_arm_measure(hw, group, wait));
	    (void) printf("trigger_measure \n");
	    CHECK(e1432_trigger_measure(hw, group, nowait));

	    if (readxdata_num>0)
	    {
		/* set up to read x memory */
		/* must be after init_measure, and arm, which disables this mode */
		bufr[0] = SRCCMDV_READ_X;
		/* (void) printf("send SRCCMDV_READ_X:  %x\n",bufr[0]); */
		xfr_error = xfr_2src(hw, src_list[0], 1, bufr,
				     E1432_SRC_DATA_MODE_RAWCMD);
	    }
	}

	if (readxdata_num>0)
	{
	    /* (void) printf("send x memory address %x\n",readxdata_addr); */
	    /* Read x memory */
	    /* write address to read from */
	    bufr[0] = readxdata_addr<<8;
	    xfr_error = xfr_2src(hw, src_list[0], 1, bufr,
				 E1432_SRC_DATA_MODE_RAWDATA);

	    /* read words */
	    /* ignore 1st word */
	    (void) printf("read %d words of x memory\n",readxdata_num);
	    bufr[0] = 0;
	    bufr[1] = 0;
	    xfr_error = xfr_2src(hw, src_list[0], readxdata_num+1, bufr,
				 E1432_SRC_DATA_MODE_RAWREAD);
	    for (bufrindx = 0; bufrindx < readxdata_num; bufrindx++)
	    { 
		bufr[bufrindx] = ((bufr[bufrindx+1]& 0xFFFFFF00UL)>>8) & 0x00FFFFFFUL;
		/* skip first word as it is garbage */
	    }
	    for (ln = 0; ln <(readxdata_num/4); ln++)
	    {
		lw =ln*4;
		(void)printf("    X:%x:  %6x, %6x, %6x, %6x\n",readxdata_addr+lw,bufr[0+lw],bufr[1+lw],bufr[2+lw],bufr[3+lw]);
	    }
	}

/* Wait for group block ready */
	error = 0;
	while (!error) /* Note ; */
	{
	    if(trace)
	    {
		(void) printf("checking block_available \n");
	    }
    
	    error = e1432_block_available(hw, in_group);
	    if (error < 0)
	    {
		(void) printf("e1432_block_available() failed, error: %d\n", error);
		return -1;
	    }
	    if (error > 0)
	    {
		CHECK(read_data());
		break;
	    }
	}	
    }

    /*NOTREACHED*/
    return 0;
}

/* Don't use the following, use the code in examples/detect.c instead. */

#define SEMA_MODEL_CODE		(0x0200)
#define DTYP_MODEL_CODE_MASK	(0x0FF8)
#define SUBC_MFR_ID_MASK	(0x0FFF)
#define SUBC_MFR_HP		(0x0FFF)

SHORTSIZ16 
e1432_detect_modules(int maxlist, SHORTSIZ16 * E1432list,
		     SHORTSIZ16 * modulecount)
{
    INST    sicl_id;		/* sicl id for the vxi bus */
    int     i;
    int     error;
    int     lalist[255];	/* list of all cards */
    int     type;
    int     mfg;
    int     numfound;		/* number of E1432s found */
    short  *baseaddr;		/* pointer to sicl space */
    volatile short *reg;	/* pointer to sicl space */

    /* open sicl interface */
    sicl_id = iopen("vxi");
    if (!sicl_id)
    {
	printf("iopen failed\n");
	exit(1);
    }
    /* get pointer to vme bus */
    baseaddr = (short *) (void *) imap(sicl_id, I_MAP_A16, 0, 0, 0);

    /* fill servant list, -1 for empty entries */
    error = ivxiservants(sicl_id, 255, lalist);
    if (error)
    {
	printf("ivxiservants failed: %d\n", error);
	exit(1);
    }

#if 0
    /* show found devices */
    printf("Devices at:");
    for (i = 0; lalist[i] >= 0; i++)
	printf(" %d", lalist[i]);
    printf("\n");
#endif

    /* check each board, copy any E1432s to E1432list */
    numfound = 0;
    for (i = 0; i < 255; i++)
    {
	if (lalist[i] >= 0)
	{
	    /* calculate device type reg address */
	    reg = (volatile short *) (void *) ((char *) baseaddr
	       + (64 * lalist[i]) + E1432_DEVICE_TYPE_REG + 0xC000);
	    /* access device type reg - if this bus faults, all is
	       lost */
	    type = *reg;
	    type = (unsigned) type;
	    /* calculate mfg reg address */
	    reg = (volatile short *) (void *) ((char *) baseaddr
			+ (64 * lalist[i]) + E1432_ID_REG + 0xC000);
	    /* access id reg - if this bus faults, all is lost */
	    mfg = *reg;
	    mfg = (unsigned) mfg;
	    /* test for E1432 */
	    if (((type & DTYP_MODEL_CODE_MASK) == SEMA_MODEL_CODE) &&
		((mfg & SUBC_MFR_ID_MASK) == SUBC_MFR_HP) &&
		(numfound < maxlist))

	    {
		E1432list[numfound] = lalist[i];
		numfound++;
	    }
	}
	else
	    break;		/* done if found -1 */
    }
    *modulecount = numfound;

#if 0
    /* show found boards */
    printf("Found %d E1432%s at:", numfound, (numfound > 1 ? "s" : ""));
    for (i = 0; i < numfound; i++)
    {
	printf(" %d", E1432list[i]);

    }
    printf("\n");
#endif

    return 0;
}
